﻿' 版权所有 (C) Microsoft Corporation。保留所有权利。
Public Class MainForm

    Private counter As PerformanceCounter


    ''' <summary>
    ''' 此子例程递减自定义计数器的值。它
    ''' 只在所选计数器是自定义计数器时才会
    ''' 执行 -- 只有自定义计数器才能将其
    ''' ReadOnly 属性设置为 False。 
    ''' </summary>
    Private Sub btnDecrementCounter_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnDecrementCounter.Click
        ' 仅当所选计数器是自定义计数器
        ' 时，此按钮才会启用，因此我们可以将只读属性设置为
        ' False，然后使用该属性。同样，请使用 Try-Catch...
        Try
            ' 将 ReadOnly 设置为 False，以便能够操作计数器
            counter.ReadOnly = False

            ' 仅当计数器的值大于 0 时才递减计数器
            ' 虽然这在技术上不是必需的，但是在逻辑上
            ' 是有意义的。
            If counter.RawValue > 0 Then
                counter.Decrement()
                ToolStripStatusLabel1.Text = ""
            Else
                ' 向用户显示状态。
                ToolStripStatusLabel1.Text = "Counter is already zero."
            End If
        Catch exc As Exception
            ' 如果引发了异常，说明
            ' 未能递减计数器。
            ToolStripStatusLabel1.Text = "Could not decrement counter."
        End Try

    End Sub

    ''' <summary>
    ''' 此子例程递增自定义计数器的值。它
    ''' 只在所选计数器是自定义计数器时才会
    ''' 执行 -- 只有自定义计数器才能将其
    ''' ReadOnly 属性设置为 False。 
    ''' </summary>
    Private Sub btnIncrementCounter_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnIncrementCounter.Click
        ' 此按钮只在所选计数器是自定义计数器
        ' 时才会启用，因此我们可以将只读属性设置为
        ' False，然后使用该计数器。同样，请使用 Try-Catch...
        Try
            ' 将 ReadOnly 设置为 False，以便能够操作计数器
            counter.ReadOnly = False
            counter.Increment()
            ToolStripStatusLabel1.Text = ""
        Catch
            ' 如果引发了异常，说明
            ' 未能递增计数器。
            ToolStripStatusLabel1.Text = "Could not increment counter."
        End Try
    End Sub

    ''' <summary>
    ''' 此子例程使用户可以在使用服务器资源管理器添加
    ''' 自定义 PerformanceCounter 后刷新窗体。
    ''' </summary>
    Private Sub btnRefreshCategories_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnRefreshCategories.Click
        ' 重置用户界面
        Me.cboCategories.Items.Clear()
        Me.cboCounters.Items.Clear()
        Me.counter = Nothing
        Me.txtBuiltInOrCustom.Text = ""
        Me.txtCounterHelp.Text = ""
        Me.txtCounterType.Text = ""
        Me.txtCounterValue.Text = ""
        Me.btnDecrementCounter.Enabled = False
        Me.btnIncrementCounter.Enabled = False
        Me.cboCategories.Text = ""
        Me.cboCounters.Text = ""

        ' 调用 Form_Load 事件
        Me.Form1_Load(Me, New System.EventArgs())
    End Sub

    ''' <summary>
    ''' 此事件处理程序在用户每次更改所选计数器
    ''' 类别时激发。然后它更改 cboCounters 组合框
    ''' 以反映可用于该类别的计数器。
    ''' 此例程使用在此项目中定义的
    ''' CounterDisplayItem 类。CounterDisplayItem 
    ''' 利用组合框显示项的方式 -- 如果它包含一个
    ''' 对象，则调用 ToString() 方法来填充显示。
    ''' 这非常重要，因为我们必须同时考虑 
    ''' “实例”（要监视的进程）和与该实例
    ''' 相关联的特定计数器。例如，可以计算
    ''' 在整个系统中编译的 CLR 字节数，或是仅针对
    ''' 正在运行的 .NET 程序的特定实例。
    ''' </summary>
    Private Sub cboCategories_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles cboCategories.SelectedIndexChanged

        Dim category As PerformanceCounterCategory
        Dim counters As New ArrayList()
        Dim thisCounter As PerformanceCounter
        Dim counterNames() As String

        If cboCategories.SelectedIndex <> -1 Then
            ' 用计数器名称填充 cboCounters。
            Try
                ' 获取可用类别。
                category = New PerformanceCounterCategory( _
                    Me.cboCategories.SelectedItem.ToString())

                ' 获取所选类别的所有可用实例。
                counterNames = category.GetInstanceNames()

                ' 如果不存在任何实例，则计数器可能是
                ' 泛型计数器，因此获取可用的计数器（它们
                ' 将没有实例值）。
                If counterNames.Length = 0 Then
                    counters.AddRange(category.GetCounters())
                Else
                    Dim i As Integer
                    For i = 0 To counterNames.Length - 1
                        counters.AddRange( _
                            category.GetCounters(counterNames(i)))
                    Next
                End If

                ' 清除 cboCounter 框并重置文本。
                Me.cboCounters.Items.Clear()
                Me.cboCounters.Text = ""

                ' 将计数器添加到 cboCounters 组合框。使用
                ' CounterDisplayItem 类确保正确显示这些
                ' 计数器，还确保存在对存储在组合框项中
                ' 的实际计数器的引用。
                For Each thisCounter In counters
                    Me.cboCounters.Items.Add(New CounterDisplayItem(thisCounter))
                Next

            Catch exc As Exception
                ' 如果程序不能列出此类别中的计数器，
                ' 则警告用户。
                MsgBox("Unable to list the Counters in this Category." + _
                    "Please select another Category.")
            End Try

        End If
    End Sub

    ''' <summary>
    ''' 此事件处理程序在用户每次更改所选计数器 
    ''' 时激发。然后它将类变量“counter”设置为计数器的
    ''' 实际值（使用存储在 cboCounters 组合框中
    ''' 的 CounterDisplayItem 对象）。进行赋值后，
    ''' 计数器的信息显示在用户界面中。
    ''' 此例程使用在此项目中定义的
    ''' CounterDisplayItem 类。
    ''' </summary>
    Private Sub cboCounters_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles cboCounters.SelectedIndexChanged

        Dim displayItem As CounterDisplayItem
        Try
            ' 获取与选择内容相关联的 CounterDisplayItem
            displayItem = CType(cboCounters.SelectedItem, CounterDisplayItem)
            ' 获取存储在 CounterDisplayItem 中的 PerformanceCounter 对象
            counter = displayItem.Counter
            ' 向用户显示有关计数器的信息
            Me.txtCounterType.Text = counter.CounterType.ToString()
            Me.txtCounterHelp.Text = counter.CounterHelp.ToString()
            ToolStripStatusLabel1.Text = ""

            ' 如果计数器是自定义计数器，则启用适当的
            ' 按钮。只有自定义项才可执行写入操作。
            ' 注意：CounterDisplayItem 显示的是
            ' 确定计数器是否是自定义计数器所必需的代码。
            If displayItem.IsCustom Then
                ' 启用“Increment”（递增）和“Decrement”（递减）按钮
                Me.txtBuiltInOrCustom.Text = "Custom"
                Me.btnDecrementCounter.Enabled = True
                Me.btnIncrementCounter.Enabled = True
            Else
                ' 禁用“Increment”（递增）和“Decrement”（递减）按钮
                Me.txtBuiltInOrCustom.Text = "Built-In"
                Me.btnDecrementCounter.Enabled = False
                Me.btnIncrementCounter.Enabled = False
            End If

        Catch exc As Exception
            ' 如果发生错误，则将类计数器设置为 Nothing。
            counter = Nothing
        End Try
    End Sub

    ''' <summary>
    ''' 此子例程用本地计算机上的所有可用类别的
    ''' 列表加载 cboCategories 组合框。它还
    ''' 启动计时器，以便每半秒更新一次用户界面。
    ''' </summary>
    Private Sub Form1_Load(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Load

        ' 用可用计数器填充 cboCounters
        Dim category As PerformanceCounterCategory
        Dim categories() As PerformanceCounterCategory
        ' 设置 myCategories 以包含所有可用类别
        categories = PerformanceCounterCategory.GetCategories()

        ' 声明一个具有正确长度的字符串数组
        Dim categoryNames(categories.Length - 1) As String
        Dim i As Integer = 0 ' 用作计数器

        ' 循环访问可用类别，将它们添加到一个
        ' 字符串数组。（这样做是为了可以
        ' 对类别进行排序。）
        For Each category In categories
            categoryNames(i) = category.CategoryName
            i += 1
        Next

        ' 对数组进行排序
        Array.Sort(categoryNames)

        ' 将数组的每个值都添加到 cboCategories 组合框。
        Dim nameString As String
        For Each nameString In categoryNames
            Me.cboCategories.Items.Add(nameString)
        Next

        ' 启动计时器
        Me.tmrUpdateUI.Interval = 500
        Me.tmrUpdateUI.Enabled = True
    End Sub

    ''' <summary>
    ''' 此事件处理程序更新用户界面中计数器
    ''' 的值。
    ''' </summary>
    Private Sub tmrUpdateUI_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tmrUpdateUI.Tick
        ' 验证计数器是否存在
        ' 如果存在，则获取其类型和值。（为防万一，请使用 Try-Catch。）
        Try
            If Not counter Is Nothing Then
                Me.txtCounterValue.Text = counter.NextValue().ToString()
            End If
        Catch exc As Exception
            Me.txtCounterValue.Text = ""
        End Try

    End Sub


    Private Sub exitToolStripMenuItem_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles exitToolStripMenuItem.Click
        Me.Close()
    End Sub
End Class